"""
BSD 2-Clause License

Copyright (c) 2024, Centre Borelli
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""
from typing import Optional

import numpy as np
from delaunay_unwrap.linprog import get_graph_structure as get_graph_structure_lp
from delaunay_unwrap.linprog import (
    solve_integrate_on_graph as solve_integrate_on_graph_lp,
)
from delaunay_unwrap.mcf import get_graph_structure as get_graph_structure_mcf
from delaunay_unwrap.mcf import solve_integrate_on_graph as solve_integrate_on_graph_mcf
from delaunay_unwrap.utils import (
    _E,
    array_and_edges_to_dict_per_edge,
    edges_to_start_end,
)
from numpy.typing import NDArray


def gradient_noise(
    num_edges: int, prop: float, intensity: float, sigma: float
) -> NDArray[np.float64]:
    bern = np.random.binomial(1, prop, num_edges)
    noise = np.random.normal(0, sigma, num_edges)
    mask = bern == 1
    num_impulse = mask.sum()
    impulse_noise = (
        2 * np.random.randint(low=0, high=2, size=num_impulse) - 1
    ) * intensity
    noise[mask] += impulse_noise
    return noise


def demo_integrate_delaunay(
    pts: NDArray[np.int64],
    values: NDArray[np.float64],
    k: int,
    prop: float,
    intensity: float,
    sigma: float,
    useSmallBasis: bool = True,
    useMCF: bool = True,
    use_weights: bool = False,
    alpha: float = 2,
    integration_ref_id: np.int32 = np.int32(0),
) -> NDArray[np.float64]:
    """
    This function is for demo purposes and must not be used otherwise.
    It computes true gradients on a redundant delaunay graph, corrupts them,
    then retrieves the corrected integrated image after gradient correction.
    """
    assert len(values) == len(pts), """Problem: values and pts 
                                must have the same length"""

    if k == 0 and useMCF:
        use_mcf = True
    else:
        use_mcf = False

    if use_mcf:
        graph_structure_mcf = get_graph_structure_mcf(pts, integration_ref_id)
        sorted_half_edges = list(graph_structure_mcf.sorted_half_edges)
    else:
        graph_structure_lp = get_graph_structure_lp(
            pts, k, useSmallBasis=useSmallBasis, integration_ref_id=integration_ref_id
        )
        sorted_half_edges = list(graph_structure_lp.sorted_half_edges)

    x, y = edges_to_start_end(sorted_half_edges)

    true_grad = values[y] - values[x]

    noise = gradient_noise(len(true_grad), prop, intensity, sigma)

    approximate_grad = true_grad + noise

    if use_weights:
        weights = 1 / np.abs((noise + 1e-3)) ** alpha

        # remove outliers
        minperc = np.percentile(weights, 1)
        maxperc = np.percentile(weights, 99)
        weights = np.clip(weights, minperc, maxperc)
        # max weight will be 1
        weights /= maxperc
        confidence_weight_per_edge: Optional[
            dict[_E, float]
        ] = array_and_edges_to_dict_per_edge(weights, sorted_half_edges)
    else:
        confidence_weight_per_edge = None

    if use_mcf:
        result = solve_integrate_on_graph_mcf(
            graph_structure_mcf,
            array_and_edges_to_dict_per_edge(approximate_grad, sorted_half_edges),
            confidence_weight_per_edge=confidence_weight_per_edge,
        )
    else:
        result = solve_integrate_on_graph_lp(
            graph_structure_lp,
            array_and_edges_to_dict_per_edge(approximate_grad, sorted_half_edges),
            confidence_weight_per_edge=confidence_weight_per_edge,
        )

    return result
