"""
BSD 2-Clause License

Copyright (c) 2024, Centre Borelli
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""
from typing import Optional

import numpy as np
from delaunay_unwrap import linprog, mcf
from numpy.typing import NDArray


# Unwraps data using either linear programming (algorithm 1) or minimum-cost flow (algorithm 5)
def unwrapk(
    pts: NDArray[np.int64],
    wrapped: NDArray[np.float64],
    k: int,
    h: float = np.pi,
    b: Optional[float] = None,
    useMCF: bool = True,
    useSmallBasis: bool = True,
    integration_ref_id: np.int32 = np.int32(0),
) -> NDArray[np.float64]:
    """


    Parameters
    ----------
    pts : NDArray[np.int64]
        (N, 2) position of points in the 2D cartesian coordinate system.
    wrapped : NDArray[np.float64]
        (N,) wrapped value associated to each point.
    k : int
        Redundancy level >= 0. 0 indicates normal Delaunay.
    h : float, optional
        Wrapping half modulus, i.e. we expect |wrapped|<= h. The default is np.pi.
    b : Optional[float], optional
        bounds for the true unwrapped gradients. The default is None.
    useMCF : bool, optional
        Wether to use the MCF algorithm, ignored if k > 0. The default is True.
    useSmallBasis : bool, optional
        Wether to use a small basis computation for linear programming. Ignored if useMCF=True and k=0. The default is True.
    integration_ref_id: np.int32, optional
        Reference node id, which will have value zero and from which we integrate gradients. The default is 0.

    Returns
    -------
    unw : NDArray[np.float64]
        (N,) unwrapped values.

    """
    if k == 0 and useMCF:
        unw = mcf.unwrap_delaunay_MCF_ts(pts, [wrapped], h, b, integration_ref_id)[0]
    else:
        unw = linprog.unwrap_delaunay_linprog_ts(
            pts, [wrapped], k, h, b, useSmallBasis, integration_ref_id
        )[0]

    return unw
