"""
BSD 2-Clause License

Copyright (c) 2024, Centre Borelli
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""
import logging
import os
import sys
import time

import fire
import imageio
import matplotlib.pyplot as plt
import numpy as np
import tifffile
from delaunay_unwrap.integration_demo import demo_integrate_delaunay
from delaunay_unwrap.unwrap import unwrapk

# Start of the demo

ACCEPTED_FORMATS = [".tiff", ".tif", ".npy", ".png", ".jpg"]


class UnrecognizedExtension(IOError):
    pass


class WrongType(TypeError):
    pass


class UnrecognizedImgShape(IOError):
    pass


logging.basicConfig(
    format="%(levelname)s:%(message)s",
    level=logging.INFO,
    handlers=[logging.FileHandler("loginfo.log"), logging.StreamHandler(sys.stdout)],
)


def ipol_error(msg):
    """Write error message to demo_failure.txt and exit with code 0"""
    logging.critical(msg)
    sys.exit(0)


def get_ext(file_path):
    ext = os.path.splitext(file_path)[1]
    return ext


def load_file(path: str, ext: str):
    u = read_img_with_arbitrary_ext(path, ext)

    if np.iscomplexobj(u):
        raise WrongType("Image data is complex, expecting real data")
    return u


def read_img_with_arbitrary_ext(file_path, ext=None):
    """
    Read a file with possibly different extensions. The extension can be passed
    as argument. Then the reader will adapt to the passed extension (either numpy or tif image).
    it the extension is not passed, we will infer it from the file path.

    Parameters
    ----------
    file_path : str
        path of the image file to be opened. Its extension can differ from ext, in case
        it has been rename, for example img.data could be a tif img that has been renamed.
    ext : str, optional
        extension of the file (origial, before potential rename). The default is None.

    Returns
    -------
    np.ndarray
        image array.

    """
    if ext is None:
        ext = get_ext(file_path)
    if ext not in ACCEPTED_FORMATS:
        msg = f"Unknown file extenstion {ext}, accepted formats are .tif, .tiff, .npy"
        raise UnrecognizedExtension(msg)

    if ext == ".npy":
        img = np.load(file_path).astype(np.float64)

    elif ext in [".tif", ".tiff"]:
        img = tifffile.imread(file_path).astype(np.float64)

    elif ext in [".png", ".jpg"]:
        img = imageio.imread(file_path).astype(np.float64)

    s = img.shape

    if len(s) == 3:
        # Grayscale but for some reason a 3rd dimension is here
        if s[2] == 1:
            img = img[:, :, 0]
        # rgb
        elif s[2] == 3:
            img = np.mean(img, axis=2)
        # rgba
        elif s[2] == 4:
            img = np.mean(img[:, :, :3], axis=2)
        else:
            raise UnrecognizedImgShape(
                f"Allowed number of channels is 1 or 3 or 4, got {s[2]}"
            )
    if len(img.shape) != 2:
        raise UnrecognizedImgShape("Unable to read or convert img into 2d array")

    return img


def main(
    phi_path,
    phi_orig_path,
    k=0,
    h=np.pi,
    b=None,
    useMCF="true",
    useSmallBasis="true",
    integration_mode="false",
    prop=0.25,
    intensity=100,
    sigma=20,
    use_weights="false",
):
    ext = get_ext(phi_orig_path)

    try:
        phi = load_file(phi_path, ext)
    except Exception as e:
        ipol_error(str(e))

    nanmask = np.isnan(phi)
    ps_mask = ~nanmask
    ps_mode = np.any(nanmask)

    if ps_mode:
        ps_rows, ps_cols = np.nonzero(ps_mask)
    else:
        height, width = phi.shape
        col_mesh, row_mesh = np.meshgrid(np.arange(width), np.arange(height))
        ps_cols = col_mesh.ravel()
        ps_rows = row_mesh.ravel()

    pts = [[ps_cols[i], ps_rows[i]] for i in range(len(ps_rows))]

    if integration_mode == "false":
        phi = (phi + h) % (2 * h) - h
        phi_ps = phi[ps_rows, ps_cols]
        start = time.time()
        result = unwrapk(
            pts, phi_ps, k, h, b, useMCF == "true", useSmallBasis == "true"
        )
        print("Total time for unwrapping : ", time.time() - start)
    else:
        phi_ps = phi[ps_rows, ps_cols]
        start = time.time()
        result = demo_integrate_delaunay(
            pts,
            phi_ps,
            k,
            prop,
            intensity,
            sigma,
            useSmallBasis == "true",
            useMCF == "true",
            use_weights == "true",
        )
        print("Total time for integration : ", time.time() - start)

    if ps_mode:
        result_raster = np.full(phi.shape, np.nan)
        result_raster[ps_rows, ps_cols] = result
    else:
        result_raster = result.reshape(phi.shape)

    if ext in [".png", ".jpg"]:
        cmap = "gray"
    else:
        cmap = "jet"

    if ps_mode:
        plt.figure()
        plt.gca().invert_yaxis()
        plt.scatter(ps_cols, ps_rows, c=phi_ps, cmap=cmap, s=2)
        plt.colorbar()
        plt.savefig("input_img.png", dpi=100)
        plt.close()

        plt.figure()
        plt.gca().invert_yaxis()
        plt.scatter(ps_cols, ps_rows, c=result, cmap=cmap, s=2)
        plt.colorbar()
        plt.savefig("output_img.png", dpi=100)
        plt.close()
    else:
        fig = plt.figure()
        plt.imshow(phi, cmap=cmap)
        plt.colorbar()
        fig.tight_layout()
        plt.savefig("input_img.png", dpi=100)
        plt.close()

        fig = plt.figure()
        plt.imshow(result_raster, cmap=cmap)
        plt.colorbar()
        fig.tight_layout()
        plt.savefig("output_img.png", dpi=100)
        plt.close()

    tifffile.imwrite("result_raster.tif", result_raster)


if __name__ == "__main__":
    fire.Fire(main)
