#!/usr/bin/env/python3
''' <This program precomputes weights used in the IPOL demo "Thin-Plate Splines on the Sphere.>
    Copyright (C) 2022 Max Dunitz

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
'''

## the following program integrates ||x-x_i||^2*log||x-x_i||
## over [0,pi]x[0,2*pi], for central points x_i in [0,pi]x[0,2*pi]

from scipy.integrate import dblquad
from scipy.special import xlogy
import numpy as np

# the Green's function of the Laplacian on R^2
# our "representer of evaluation" 
# (up to a constant, modulo something in H_0)
# at (x_i, y_i), evaluated at arbitrary (x,y)
def k_(x,y,x_i,y_i):
    norm_sq = (x-x_i)**2 + (y-y_i)**2
    return xlogy(norm_sq, norm_sq)

# choose center points 
lons = np.linspace(0, 6.283, 6284)
lats = np.linspace(0,3.142,3143)
lats_, lons_ = np.meshgrid(lats, lons, indexing='ij')

# compute the integrals to the given precision
vals = np.zeros(lats_.shape)
def fvec(la, lo):
    f = lambda y, x: k_(x,y,lo,la)
    return dblquad(f, 0, 2*np.pi, 0, np.pi, epsabs=1e-5, epsrel=1e-5)[0]
vfunc = np.vectorize(fvec)
vals = vfunc(lats_, lons_)/(4*np.pi)

# save the result
np.save("int_vals.npy", vals)

